<?php
// public/dons-parcelle.php
ini_set('display_errors', 1);
error_reporting(E_ALL);

require_once '../src/config/auth.php';
$auth = new Auth();

if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

$db = (new Database())->getConnection();

// Récupérer les appels de fond
$stmt = $db->query("SELECT id, titre FROM appels_de_fond ORDER BY date_debut DESC");
$appels = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1.0" />
    <title>Souscriptions - Goshen Tabernacle</title>
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet" />
</head>
<body class="bg-gray-100 text-gray-900 font-sans">

    <!-- Sidebar -->
    <?php include 'partials/sidebar.php'; ?>

    <!-- Contenu Principal -->
    <main class="ml-64 p-6">
        <div class="max-w-6xl mx-auto">
            <h1 class="text-2xl font-bold mb-2">🎁 Souscriptions par Appel de Fond</h1>
            <p class="text-gray-600 mb-6">Suivi des engagements des membres (CDF et USD)</p>

            <!-- Message dynamique -->
            <div id="message" class="hidden p-4 mb-6 rounded bg-green-100 text-green-800"></div>

            <!-- Boutons d'appel de fond -->
            <div class="mb-6">
                <h3 class="font-bold mb-2">Sélectionner un appel de fond :</h3>
                <div class="flex flex-wrap gap-2">
                    <?php foreach ($appels as $a): ?>
                        <button 
                            onclick="chargerSouscriptions(<?= (int)$a['id'] ?>)" 
                            class="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded text-sm">
                            <?= htmlspecialchars($a['titre'], ENT_QUOTES, 'UTF-8') ?>
                        </button>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Recherche en direct -->
            <div class="mb-6">
                <input 
                    type="text" 
                    id="searchInput" 
                    placeholder="Rechercher un membre..." 
                    class="border p-3 rounded w-full"
                    onkeyup="filterSouscriptions(this.value)" />
            </div>

            <!-- Formulaire d'ajout (AJAX) -->
            <div class="bg-white p-6 rounded shadow mb-8">
                <h2 class="text-xl font-bold mb-4">➕ Nouvelle Souscription</h2>
                <form id="form-souscription" class="grid grid-cols-1 md:grid-cols-5 gap-4">
                    <input type="text" name="nom_membre" placeholder="Nom du membre" required class="border p-3 rounded" />
                    <input type="number" name="montant_cdf" placeholder="Montant CDF" step="0.01" class="border p-3 rounded" />
                    <input type="number" name="montant_usd" placeholder="Montant USD" step="0.01" class="border p-3 rounded" />
                    <select name="appel_id" required class="border p-3 rounded">
                        <option value="">Sélectionner un appel</option>
                        <?php foreach ($appels as $a): ?>
                            <option value="<?= (int)$a['id'] ?>"><?= htmlspecialchars($a['titre'], ENT_QUOTES, 'UTF-8') ?></option>
                        <?php endforeach; ?>
                    </select>
                    <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-4 py-3 rounded font-medium">
                        <i class="fas fa-plus"></i> Ajouter
                    </button>
                </form>
            </div>

            <!-- Liste des souscriptions -->
            <div class="bg-white p-6 rounded shadow">
                <h2 class="text-xl font-bold mb-4">📋 Liste des Souscriptions</h2>
                <table class="w-full border-collapse text-sm">
                    <thead class="bg-gray-100">
                        <tr>
                            <th class="border p-2 text-left">Appel</th>
                            <th class="border p-2 text-left">Membre</th>
                            <th class="border p-2 text-right">Engagé (CDF)</th>
                            <th class="border p-2 text-right">Payé (CDF)</th>
                            <th class="border p-2 text-right">Reste (CDF)</th>
                            <th class="border p-2 text-right">Engagé (USD)</th>
                            <th class="border p-2 text-right">Payé (USD)</th>
                            <th class="border p-2 text-right">Reste (USD)</th>
                            <th class="border p-2 text-center">Statut</th>
                            <th class="border p-2 text-center">Paiement Partiel</th>
                        </tr>
                    </thead>
                    <tbody id="liste-souscriptions">
                        <tr>
                            <td colspan="10" class="text-center text-gray-500">
                                Sélectionnez un appel de fond pour charger les données
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <!-- Script AJAX -->
    <script>
        let currentAppelId = null;

        // Afficher un message temporaire
        function showMessage(text, type = 'success') {
            const msg = document.getElementById('message');
            msg.textContent = text;
            msg.className = `p-4 mb-6 rounded ${type === 'error' ? 'bg-red-100 text-red-800' : 'bg-green-100 text-green-800'}`;
            msg.style.display = 'block';
            setTimeout(() => msg.style.display = 'none', 5000);
        }

        // Charger les souscriptions via AJAX
function chargerSouscriptions(appelId) {
    if (!appelId || appelId <= 0) {
        showMessage("ID d'appel invalide", "error");
        return;
    }

    const tbody = document.getElementById('liste-souscriptions');
    tbody.innerHTML = '<tr><td colspan="10" class="text-center text-gray-500">Chargement...</td></tr>';

    fetch(`api/souscriptions.php?appel_id=${appelId}`)
        .then(response => {
            if (!response.ok) throw new Error('Erreur réseau');
            return response.json();
        })
        .then(data => {
            // ✅ Vérifier si erreur
            if (data.error) {
                tbody.innerHTML = `<tr><td colspan="10" class="text-center text-red-600">${data.error}</td></tr>`;
                return;
            }

            // ✅ Vérifier s'il y a des données
            if (!data.souscriptions || data.souscriptions.length === 0) {
                const tr = document.createElement('tr');
                tr.innerHTML = '<td colspan="10" class="text-center text-gray-500">Aucune souscription</td>';
                tbody.appendChild(tr);
                return;
            }

            // ✅ Afficher les données
            afficherSouscriptions(data.souscriptions);
        })
        .catch(err => {
            console.error('Erreur AJAX:', err);
            tbody.innerHTML = `<tr><td colspan="10" class="text-center text-red-600">Échec du chargement</td></tr>`;
        });
}

// Afficher les souscriptions
function afficherSouscriptions(souscriptions) {
    const tbody = document.getElementById('liste-souscriptions');
    tbody.innerHTML = ''; // ✅ Vide le tableau

    souscriptions.forEach(s => {
        const resteCdf = Math.max(0, parseFloat(s.montant_cdf) - parseFloat(s.total_paye_cdf));
        const resteUsd = Math.max(0, parseFloat(s.montant_usd) - parseFloat(s.total_paye_usd));

        const tr = document.createElement('tr');
        tr.dataset.nom = s.nom_membre.toLowerCase();

        tr.innerHTML = `
            <td class="border p-2 font-medium">${s.appel_titre || 'Inconnu'}</td>
            <td class="border p-2">${s.nom_membre}</td>
            <td class="border p-2 text-right">${Number(s.montant_cdf).toLocaleString()}</td>
            <td class="border p-2 text-right">${Number(s.total_paye_cdf).toLocaleString()}</td>
            <td class="border p-2 text-right font-bold">${Number(resteCdf).toLocaleString()}</td>
            <td class="border p-2 text-right">${Number(s.montant_usd).toFixed(2)}</td>
            <td class="border p-2 text-right">${Number(s.total_paye_usd).toFixed(2)}</td>
            <td class="border p-2 text-right font-bold">${Number(resteUsd).toFixed(2)}</td>
            <td class="border p-2 text-center">
                <span class="px-2 py-1 rounded text-xs 
                    ${s.statut === 'complet' ? 'bg-green-200 text-green-800' : 'bg-yellow-200 text-yellow-800'}">
                    ${s.statut === 'complet' ? 'Complet' : 'En cours'}
                </span>
            </td>
            <td class="border p-2 text-center">
                <form class="inline" onsubmit="enregistrerPaiement(event, ${s.id})">
                    <div class="flex gap-1 mb-1">
                        <input type="number" name="paiement_cdf" placeholder="CDF" step="0.01" class="w-20 text-center border p-1 rounded text-xs" />
                        <input type="number" name="paiement_usd" placeholder="USD" step="0.01" class="w-20 text-center border p-1 rounded text-xs" />
                    </div>
                    <button type="submit" class="bg-blue-600 text-white px-2 py-1 rounded text-xs">
                        <i class="fas fa-hand-holding-usd"></i> Encaisser
                    </button>
                </form>
            </td>
        `;
        tbody.appendChild(tr);
    });
}

        // Fonction pour afficher les souscriptions
function afficherSouscriptions(souscriptions) {
    const tbody = document.getElementById('liste-souscriptions');
    tbody.innerHTML = ''; // Vide le tableau

    souscriptions.forEach(s => {
        const resteCdf = Math.max(0, parseFloat(s.montant_cdf) - parseFloat(s.total_paye_cdf));
        const resteUsd = Math.max(0, parseFloat(s.montant_usd) - parseFloat(s.total_paye_usd));

        const tr = document.createElement('tr');
        tr.dataset.nom = s.nom_membre.toLowerCase();

        tr.innerHTML = `
            <td class="border p-2 font-medium">${s.appel_titre || 'Appel Inconnu'}</td>
            <td class="border p-2">${s.nom_membre}</td>
            <td class="border p-2 text-right">${Number(s.montant_cdf).toLocaleString()}</td>
            <td class="border p-2 text-right">${Number(s.total_paye_cdf).toLocaleString()}</td>
            <td class="border p-2 text-right font-bold">${Number(resteCdf).toLocaleString()}</td>
            <td class="border p-2 text-right">${Number(s.montant_usd).toFixed(2)}</td>
            <td class="border p-2 text-right">${Number(s.total_paye_usd).toFixed(2)}</td>
            <td class="border p-2 text-right font-bold">${Number(resteUsd).toFixed(2)}</td>
            <td class="border p-2 text-center">
                <span class="px-2 py-1 rounded text-xs 
                    ${s.statut === 'complet' ? 'bg-green-200 text-green-800' : 'bg-yellow-200 text-yellow-800'}">
                    ${s.statut === 'complet' ? 'Complet' : 'En cours'}
                </span>
            </td>
            <td class="border p-2 text-center">
                <form class="inline" onsubmit="enregistrerPaiement(event, ${s.id})">
                    <div class="flex gap-1 mb-1">
                        <input type="number" name="paiement_cdf" placeholder="CDF" step="0.01" class="w-20 text-center border p-1 rounded text-xs" />
                        <input type="number" name="paiement_usd" placeholder="USD" step="0.01" class="w-20 text-center border p-1 rounded text-xs" />
                    </div>
                    <button type="submit" class="bg-blue-600 text-white px-2 py-1 rounded text-xs">
                        <i class="fas fa-hand-holding-usd"></i> Encaisser
                    </button>
                </form>
            </td>
        `;
        tbody.appendChild(tr);
    });
}

        // Recherche en direct
        function filterSouscriptions(query) {
            query = query.toLowerCase().trim();
            document.querySelectorAll('#liste-souscriptions tr').forEach(tr => {
                const nom = tr.dataset.nom || '';
                tr.style.display = nom.includes(query) ? '' : 'none';
            });
        }

        // Ajouter une souscription (AJAX)
        document.getElementById('form-souscription').addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);

            fetch('api/ajouter-souscription.php', {
                method: 'POST',
                body: JSON.stringify(data),
                headers: { 'Content-Type': 'application/json' }
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    showMessage(data.message);
                    const tr = document.createElement('tr');
                    tr.dataset.nom = data.souscription.nom_membre.toLowerCase();
                    tr.innerHTML = `
                        <td class="border p-2 font-medium">\${data.souscription.appel_titre || 'Inconnu'}</td>
                        <td class="border p-2">\${data.souscription.nom_membre}</td>
                        <td class="border p-2 text-right">\${Number(data.souscription.montant_cdf).toLocaleString()}</td>
                        <td class="border p-2 text-right">0</td>
                        <td class="border p-2 text-right font-bold">\${Number(data.souscription.reste_cdf).toLocaleString()}</td>
                        <td class="border p-2 text-right">\${Number(data.souscription.montant_usd).toFixed(2)}</td>
                        <td class="border p-2 text-right">0.00</td>
                        <td class="border p-2 text-right font-bold">\${Number(data.souscription.reste_usd).toFixed(2)}</td>
                        <td class="border p-2 text-center">
                            <span class="px-2 py-1 rounded text-xs bg-yellow-200 text-yellow-800">En cours</span>
                        </td>
                        <td>...</td>
                    `;
                    document.getElementById('liste-souscriptions').prepend(tr);
                    this.reset();
                } else {
                    showMessage(data.error, 'error');
                }
            })
            .catch(err => {
                showMessage('Erreur réseau', 'error');
                console.error(err);
            });
        });

        // Enregistrer un paiement (AJAX)
        function enregistrerPaiement(e, id) {
    e.preventDefault();
    const form = e.target;
    const cdf = parseFloat(form.paiement_cdf.value) || 0;
    const usd = parseFloat(form.paiement_usd.value) || 0;

    if (cdf <= 0 && usd <= 0) {
        showMessage('Montant invalide', 'error');
        return;
    }

    fetch('api/ajouter-paiement.php', {
        method: 'POST',
        body: JSON.stringify({ souscription_id: id, paiement_cdf: cdf, paiement_usd: usd }),
        headers: { 'Content-Type': 'application/json' }
    })
    .then(r => r.json())
    .then(data => {
        if (data.success) {
            showMessage('Paiement enregistré');
            const tr = form.closest('tr');

            // ✅ Mise à jour des cellules
            tr.querySelector('td:nth-child(4)').textContent = Number(data.solde.total_paye_cdf).toLocaleString(); // Payé CDF
            tr.querySelector('td:nth-child(5)').textContent = Number(data.solde.reste_cdf).toLocaleString();      // Reste CDF
            tr.querySelector('td:nth-child(7)').textContent = Number(data.solde.total_paye_usd).toFixed(2);      // Payé USD
            tr.querySelector('td:nth-child(8)').textContent = Number(data.solde.reste_usd).toFixed(2);           // Reste USD

            // ✅ Mise à jour du statut
            const span = tr.querySelector('td:nth-child(9) span'); // Sélectionnez le span dans le statut
            span.textContent = data.solde.statut === 'complet' ? 'Complet' : 'En cours';
            span.className = data.solde.statut === 'complet'
                ? 'px-2 py-1 rounded text-xs bg-green-200 text-green-800'
                : 'px-2 py-1 rounded text-xs bg-yellow-200 text-yellow-800';

            // ✅ Vider les champs après paiement
            form.reset();
        } else {
            showMessage(data.error || 'Erreur inconnue', 'error');
        }
    })
    .catch(err => {
        console.error('Erreur AJAX:', err);
        showMessage('Erreur réseau ou serveur', 'error');
    });
}
    </script>
</body>
</html>