<?php
// public/api/ajouter-paiement.php
header('Content-Type: application/json');

// 🔹 Activer les erreurs pour le débogage (à désactiver en production)
ini_set('display_errors', 1);
error_reporting(E_ALL);

// 🔹 Charger l'authentification
require_once '../../src/config/auth.php';

$auth = new Auth();

if (!$auth->isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['error' => 'Non connecté']);
    exit;
}

// 🔹 Connexion à la base
try {
    $db = (new Database())->getConnection();
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erreur DB: ' . $e->getMessage()]);
    exit;
}

// 🔹 Lire les données JSON
$input = file_get_contents('php://input');
if (!$input) {
    http_response_code(400);
    echo json_encode(['error' => 'Aucune donnée reçue']);
    exit;
}

$data = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    http_response_code(400);
    echo json_encode(['error' => 'JSON invalide: ' . json_last_error_msg()]);
    exit;
}

// 🔹 Extraire les données
$souscription_id = intval($data['souscription_id'] ?? 0);
$montant_cdf = floatval($data['paiement_cdf'] ?? 0);
$montant_usd = floatval($data['paiement_usd'] ?? 0);

// 🔹 Validation
if ($souscription_id <= 0 || ($montant_cdf <= 0 && $montant_usd <= 0)) {
    http_response_code(400);
    echo json_encode(['error' => 'Données invalides']);
    exit;
}

// 🔹 Transaction
try {
    $db->beginTransaction();

    // 1. Enregistrer le paiement
    $stmt = $db->prepare("
        INSERT INTO paiements_tranches (souscription_id, montant_cdf, montant_usd) 
        VALUES (?, ?, ?)
    ");
    $stmt->execute([$souscription_id, $montant_cdf, $montant_usd]);

    // 2. Calculer le nouveau solde
    $stmt = $db->prepare("
        SELECT 
            s.montant_cdf, s.montant_usd,
            COALESCE(SUM(p.montant_cdf), 0) AS total_paye_cdf,
            COALESCE(SUM(p.montant_usd), 0) AS total_paye_usd
        FROM souscriptions s
        LEFT JOIN paiements_tranches p ON s.id = p.souscription_id
        WHERE s.id = ?
        GROUP BY s.id
    ");
    $stmt->execute([$souscription_id]);
    $solde = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$solde) {
        throw new Exception("Souscription non trouvée");
    }

    // 3. Calculer le reste
    $reste_cdf = max(0, $solde['montant_cdf'] - $solde['total_paye_cdf']);
    $reste_usd = max(0, $solde['montant_usd'] - $solde['total_paye_usd']);
    $statut = ($reste_cdf <= 0 && $reste_usd <= 0) ? 'complet' : 'actif';

    // 4. Mettre à jour le statut
    $db->prepare("UPDATE souscriptions SET statut = ? WHERE id = ?")
       ->execute([$statut, $souscription_id]);

    $db->commit();

    // 5. Retourner la réponse
    echo json_encode([
        'success' => true,
        'message' => 'Paiement enregistré',
        'solde' => [
            'total_paye_cdf' => $solde['total_paye_cdf'],
            'total_paye_usd' => $solde['total_paye_usd'],
            'reste_cdf' => $reste_cdf,
            'reste_usd' => $reste_usd,
            'statut' => $statut
        ]
    ]);

} catch (Exception $e) {
    $db->rollback();
    error_log("Erreur paiement: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Erreur serveur : ' . $e->getMessage()]);
}